<?php
class TextReplacer {
    private $replacements = [];
    private $regexReplacements = [];
    private $dictionaryPath = '';
    
    public function __construct($dictionaryPath = '') {
        if ($dictionaryPath) {
            $this->setDictionaryPath($dictionaryPath);
        }
        $this->presetCommonRules();
    }
    
    public function setDictionaryPath($path) {
        if (file_exists($path)) {
            $this->dictionaryPath = $path;
            $this->loadDictionary();
        } else {
            throw new Exception("Dictionary file not found: " . $path);
        }
    }
    
    private function loadDictionary() {
        $lines = file($this->dictionaryPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
            if (strpos(trim($line), '#') === 0) continue;
            
            $parts = explode('=>', $line, 2);
            if (count($parts) === 2) {
                $search = trim($parts[0]);
                $replace = trim($parts[1]);
                
                if (preg_match('/^\/.+\/[a-zA-Z]*$/u', $search)) {
                    // 特别处理带有u修饰符的正则表达式
                    $this->addRegexReplacement($search, $replace, true);
                } else {
                    $this->addReplacement($search, $replace);
                }
            }
        }
    }
    
    public function addReplacement($search, $replace) {
        $this->replacements[$search] = $replace;
    }
    
    public function addRegexReplacement($pattern, $replace, $unicode = false) {
        // 确保正则表达式有u修饰符（如果需要）
        if ($unicode && strpos($pattern, 'u') === false) {
            $pattern .= 'u';
        }
        $this->regexReplacements[$pattern] = $replace;
    }
    
    public function addReplacements(array $rules) {
        foreach ($rules as $search => $replace) {
            if (preg_match('/^\/.+\/[a-zA-Z]*$/u', $search)) {
                $this->addRegexReplacement($search, $replace, true);
            } else {
                $this->addReplacement($search, $replace);
            }
        }
    }
    
    private function presetCommonRules() {
        $this->addReplacement('teh', 'the');
        $this->addReplacement('adn', 'and');
        $this->addReplacement('thier', 'their');
        $this->addRegexReplacement('/\b(\d+)st\b/u', '$1th');
        $this->addRegexReplacement('/\b(\d+)nd\b/u', '$1th');
        $this->addRegexReplacement('/\b(\d+)rd\b/u', '$1th');
    }
    
    public function replace($text) {
        // 确保UTF-8编码
        if (!mb_check_encoding($text, 'UTF-8')) {
            $text = mb_convert_encoding($text, 'UTF-8');
        }
        
        // 先执行正则替换（确保优先处理复杂规则）
        if (!empty($this->regexReplacements)) {
            foreach ($this->regexReplacements as $pattern => $replacement) {
                try {
                    // 特殊处理包含回调语法的替换
                    if (is_string($replacement) && preg_match('/strtolower|strtoupper/i', $replacement)) {
                        $text = preg_replace_callback($pattern, 
                            function($matches) use ($replacement) {
                                return eval('return ' . $replacement . ';');
                            }, 
                            $text);
                    } else {
                        $text = preg_replace($pattern, $replacement, $text);
                    }
                } catch (Exception $e) {
                    // 忽略单个规则错误，继续执行其他规则
                    continue;
                }
            }
        }
        
        // 然后执行简单替换
        if (!empty($this->replacements)) {
            $text = str_replace(
                array_keys($this->replacements),
                array_values($this->replacements),
                $text
            );
        }
        
        return $text;
    }
    
    public function getRules() {
        return [
            'simple' => $this->replacements,
            'regex' => $this->regexReplacements
        ];
    }
    
    public function clearRules() {
        $this->replacements = [];
        $this->regexReplacements = [];
    }
}