<?php
// remote_content_api.php - 基于RemoteDB的远程内容API

require_once 'RemoteDB.php';

// 配置
define('DB_FILE', 'content_ai.db');
define('API_URL', 'http://pan.zgyzty.com/api/AIGC/renzaocao/');
define('ITEMS_PER_PAGE', 10);

// 初始化远程数据库连接$db = new RemoteDB('./test.d3','https://atm-tkd.sakura.ne.jp/pcspe/access/');
function initRemoteDB() {
    return new RemoteDB(DB_FILE, API_URL);
}

// 获取内容列表
function getContents($page = 1, $keyword = '', $categoryId = null) {
    $db = initRemoteDB();
    $offset = ($page - 1) * ITEMS_PER_PAGE;
    
    // 构建查询条件
    $where = [];
    $params = [];
    
    if (!empty($keyword)) {
        $where[] = "(gc.question LIKE ? OR gc.answer LIKE ?)";
        $params[] = "%$keyword%";
        $params[] = "%$keyword%";
    }
    
    if ($categoryId) {
        $where[] = "gc.category_id = ?";
        $params[] = $categoryId;
    }
    
    $whereClause = $where ? "WHERE " . implode(" AND ", $where) : "";
    
    // 主查询
    $query = "
        SELECT gc.id, gc.question, gc.answer, gc.created_at, 
               c.id as category_id, c.name as category_name 
        FROM generated_content gc
        LEFT JOIN categories c ON gc.category_id = c.id
        $whereClause
        ORDER BY gc.created_at DESC 
        LIMIT " . ITEMS_PER_PAGE . " OFFSET $offset
    ";
    
    $contents = $db->query($query, $params) ?: [];
    
    // 获取总数
    $countQuery = "SELECT COUNT(*) as total FROM generated_content gc $whereClause";
    $total = $db->query($countQuery, $params)[0]['total'] ?? 0;
    
    // 获取分类列表
    $categories = $db->query("SELECT id, name FROM categories ORDER BY name") ?: [];
    
    return [
        'contents' => $contents,
        'total' => $total,
        'total_pages' => ceil($total / ITEMS_PER_PAGE),
        'current_page' => $page,
        'categories' => $categories,
        'current_category' => $categoryId,
        'keyword' => $keyword
    ];
}

// 获取单个内容详情
function getContent($id) {
    $db = initRemoteDB();
    $query = "
        SELECT gc.id, gc.question, gc.answer, gc.created_at, 
               c.id as category_id, c.name as category_name 
        FROM generated_content gc
        LEFT JOIN categories c ON gc.category_id = c.id
        WHERE gc.id = ?
    ";
    $result = $db->query($query, [$id]);
    return $result[0] ?? null;
}

// 输出JSON响应
function jsonResponse($data) {
    header('Content-Type: application/json');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// 输出HTML列表页
function htmlListPage($data) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>内容列表 - 远程数据库</title>
        <link href="https://cdn.bootcdn.net/ajax/libs/picocss/2.1.0/pico.css" rel="stylesheet">
        <style>
            .pagination { margin: 20px 0; }
            .pagination a, .pagination strong { 
                display: inline-block; 
                padding: 5px 10px; 
                margin: 0 2px; 
                border: 1px solid #ddd; 
                text-decoration: none;
            }
            .pagination a:hover { background: #eee; }
            .pagination strong { background: #0073aa; color: white; }
            .search-form { margin-bottom: 20px; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>内容列表 - 远程数据库</h1>
            
            <form class="search-form" method="get">
                <input type="hidden" name="action" value="list">
                <div class="grid">
                    <div>
                        <input type="text" name="keyword" placeholder="搜索关键词" value="<?= htmlspecialchars($data['keyword']) ?>">
                    </div>
                    <div>
                        <select name="category">
                            <option value="">所有分类</option>
                            <?php foreach ($data['categories'] as $cat): ?>
                                <option value="<?= $cat['id'] ?>" <?= $data['current_category'] == $cat['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($cat['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <button type="submit">搜索</button>
                    </div>
                </div>
            </form>
            
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>分类</th>
                        <th>问题</th>
                        <th>创建时间</th>
                        <th>操作</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($data['contents'] as $item): ?>
                        <tr>
                            <td><?= htmlspecialchars($item['id']) ?></td>
                            <td><?= htmlspecialchars($item['category_name']) ?></td>
                            <td><?= htmlspecialchars(mb_substr($item['question'], 0, 50, 'UTF-8')) ?></td>
                            <td><?= htmlspecialchars($item['created_at']) ?></td>
                            <td>
                                <a href="?action=view&id=<?= $item['id'] ?>">查看</a>
                                <a href="?action=view&id=<?= $item['id'] ?>&type=json">JSON</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <?php if ($data['total_pages'] > 1): ?>
                <div class="pagination">
                    <?php if ($data['current_page'] > 1): ?>
                        <a href="?action=list&page=1&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">首页</a>
                        <a href="?action=list&page=<?= $data['current_page']-1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">上一页</a>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $data['total_pages']; $i++): ?>
                        <?php if ($i == $data['current_page']): ?>
                            <strong><?= $i ?></strong>
                        <?php else: ?>
                            <a href="?action=list&page=<?= $i ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>"><?= $i ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <?php if ($data['current_page'] < $data['total_pages']): ?>
                        <a href="?action=list&page=<?= $data['current_page']+1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">下一页</a>
                        <a href="?action=list&page=<?= $data['total_pages'] ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">末页</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </body>
    </html>
    <?php
}

// 输出HTML详情页
function htmlViewPage($content) {
    if (!$content) {
        echo "<p>内容不存在</p>";
        echo "<a href='?action=list'>返回列表</a>";
        return;
    }
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title><?= htmlspecialchars(mb_substr($content['question'], 0, 20, 'UTF-8')) ?> - 远程数据库</title>
        <link href="https://cdn.bootcdn.net/ajax/libs/picocss/2.1.0/pico.css" rel="stylesheet">
    </head>
    <body>
        <div class="container">
            <h1><?= htmlspecialchars($content['question']) ?></h1>
            <p>分类: <?= htmlspecialchars($content['category_name']) ?></p>
            <p>创建时间: <?= htmlspecialchars($content['created_at']) ?></p>
            
            <div style="border:1px solid #ccc; padding:15px; margin:15px 0;">
                <?= nl2br(htmlspecialchars($content['answer'])) ?>
            </div>
            
            <div>
                <a href="?action=list">返回列表</a> | 
                <a href="?action=view&id=<?= $content['id'] ?>&type=json">查看JSON格式</a>
            </div>
        </div>
    </body>
    </html>
    <?php
}

// 主控制器
function handleRequest() {
    $action = $_GET['action'] ?? 'list';
    $outputType = $_GET['type'] ?? 'html';
    
    try {
        switch ($action) {
            case 'list':
                $page = max(1, intval($_GET['page'] ?? 1));
                $keyword = trim($_GET['keyword'] ?? '');
                $categoryId = $_GET['category'] ? intval($_GET['category']) : null;
                
                $data = getContents($page, $keyword, $categoryId);
                
                if ($outputType === 'json') {
                    jsonResponse($data);
                } else {
                    htmlListPage($data);
                }
                break;
                
            case 'view':
                if (!isset($_GET['id'])) {
                    throw new Exception("缺少ID参数");
                }
                
                $id = intval($_GET['id']);
                $content = getContent($id);
                
                if ($outputType === 'json') {
                    jsonResponse($content ?: ['error' => 'Content not found']);
                } else {
                    htmlViewPage($content);
                }
                break;
                
            default:
                throw new Exception("无效的操作");
        }
    } catch (Exception $e) {
        if ($outputType === 'json') {
            jsonResponse(['error' => $e->getMessage()]);
        } else {
            echo "<p>错误: " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    }
}

// 启动应用
handleRequest();
?>