<?php
/**
 * Markdown解析器与TailwindCSS生成器
 * 兼容PHP5.2+
 */

// 1. Markdown解析函数 (简化版)
function parseMarkdown($markdown) {
    $lines = explode("\n", $markdown);
    $contentData = array(
        'title' => '',
        'description' => '',
        'sections' => array(),
        'meta' => array(
            'keywords' => '',
            'author' => 'Markdown解析器'
        )
    );
    
    $currentSection = array(
        'title' => '内容',
        'type' => 'section',
        'level' => 2,
        'content' => array()
    );
    
    $inList = false;
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        // 提取主标题
        if (empty($contentData['title']) && preg_match('/^#\s+(.+)/', $line, $matches)) {
            $contentData['title'] = $matches[1];
            continue;
        }
        
        // 提取副标题作为描述
        if (empty($contentData['description']) && preg_match('/^##\s+(.+)/', $line, $matches)) {
            $contentData['description'] = $matches[1];
            continue;
        }
        
        // 处理章节标题
        if (preg_match('/^#{1,3}\s+(.+)/', $line, $matches)) {
            // 保存前一章节（如果已有内容）
            if (!empty($currentSection['content'])) {
                $contentData['sections'][] = $currentSection;
            }
            
            // 确定标题级别
            $level = strlen(explode(' ', $line)[0]);
            
            // 创建新章节
            $currentSection = array(
                'title' => $matches[1],
                'type' => 'section',
                'level' => $level,
                'content' => array()
            );
            continue;
        }
        
        // 处理列表
        if (preg_match('/^\*\s+(.+)/', $line, $matches)) {
            if (!$inList) {
                if (empty($currentSection['content']) || end($currentSection['content'])['type'] !== 'list') {
                    $currentSection['content'][] = array(
                        'type' => 'list',
                        'items' => array()
                    );
                }
                $inList = true;
            }
            $listItem = trim($matches[1]);
            $lastIndex = count($currentSection['content']) - 1;
            $currentSection['content'][$lastIndex]['items'][] = $listItem;
            continue;
        } else {
            $inList = false;
        }
        
        // 处理强调文本
        if (preg_match('/\*\*(.+?)\*\*/', $line, $matches)) {
            $line = str_replace($matches[0], '<strong>'.$matches[1].'</strong>', $line);
        }
        if (preg_match('/\*(.+?)\*/', $line, $matches)) {
            $line = str_replace($matches[0], '<em>'.$matches[1].'</em>', $line);
        }
        
        // 处理普通段落
        if (!empty($line)) {
            $currentSection['content'][] = array(
                'type' => 'paragraph',
                'text' => $line
            );
        }
    }
    
    // 添加最后一个章节（如果已有内容）
    if (!empty($currentSection['content'])) {
        $contentData['sections'][] = $currentSection;
    }
    
    // 如果没有提取到标题，使用第一段作为标题
    if (empty($contentData['title']) && !empty($contentData['sections'])) {
        $firstSection = $contentData['sections'][0];
        if (!empty($firstSection['content'])) {
            $firstContent = $firstSection['content'][0];
            if ($firstContent['type'] === 'paragraph') {
                $contentData['title'] = mb_substr($firstContent['text'], 0, 30, 'UTF-8');
                if (mb_strlen($firstContent['text'], 'UTF-8') > 30) {
                    $contentData['title'] .= '...';
                }
            }
        }
    }
    
    return $contentData;
}

// 2. 从文件读取Markdown内容
$markdownContent = file_get_contents('test.md');

// 3. 解析Markdown
$contentData = parseMarkdown($markdownContent);

// 4. HTML生成函数
function generateHtmlPage($data) {
    // 检测PHP版本以决定使用哪种JSON编码方式
    $jsonData = function_exists('json_encode') 
        ? json_encode($data) 
        : custom_json_encode($data);
    
    // 开始输出HTML
    $html = '<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($data['title']) . '</title>
    <meta name="description" content="' . htmlspecialchars($data['description']) . '">
    <meta name="keywords" content="' . htmlspecialchars(isset($data['meta']['keywords']) ? $data['meta']['keywords'] : '' ) . '">
    <meta name="author" content="' . htmlspecialchars(isset($data['meta']['author']) ? $data['meta']['author'] : '' ) . '">
    
    <!-- 引入TailwindCSS CDN -->
    <script src="https://cdn.tailwindcss.com/3.4.1"></script>
    
    <!-- 自定义样式 -->
    <style>
        .prose a { color: #3b82f6; text-decoration: underline; }
        .prose a:hover { color: #2563eb; }
        .content-block { transition: all 0.3s ease; }
        .content-block:hover { transform: translateY(-2px); box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1); }
    </style>
</head>
<body class="bg-gray-50 font-sans leading-relaxed text-gray-800">
    
    <!-- 导航栏 -->
    <nav class="bg-white shadow-sm sticky top-0 z-10">
        <div class="container mx-auto px-4">
            <div class="flex justify-between items-center py-4">
                <div class="text-xl font-bold text-blue-600">
                    ' . htmlspecialchars($data['title']) . '
                </div>
                <div class="hidden md:flex space-x-6">
                    <a href="#content" class="text-gray-600 hover:text-blue-600">内容</a>
                    <a href="#pricing" class="text-gray-600 hover:text-blue-600">价格</a>
                    <a href="#factors" class="text-gray-600 hover:text-blue-600">影响因素</a>
                </div>
                <button id="menu-button" class="md:hidden text-gray-600 hover:text-blue-600">
                    <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"></path>
                    </svg>
                </button>
            </div>
        </div>
        <!-- 移动菜单 -->
        <div id="mobile-menu" class="hidden md:hidden bg-white border-t">
            <div class="container mx-auto px-4 py-2 flex flex-col space-y-2">
                <a href="#content" class="py-2 text-gray-600 hover:text-blue-600">内容</a>
                <a href="#pricing" class="py-2 text-gray-600 hover:text-blue-600">价格</a>
                <a href="#factors" class="py-2 text-gray-600 hover:text-blue-600">影响因素</a>
            </div>
        </div>
    </nav>
    
    <!-- 主要内容 -->
    <main class="container mx-auto px-4 py-8 max-w-4xl">
        <!-- 页面标题 -->
        <header class="mb-12 text-center">
            <h1 class="text-4xl font-bold text-gray-900 mb-4">' . htmlspecialchars($data['title']) . '</h1>
            <p class="text-xl text-gray-600">' . htmlspecialchars($data['description']) . '</p>
        </header>
        
        <!-- 动态生成内容 -->';
        
    // 生成各个内容区块
    foreach ($data['sections'] as $section) {
        $html .= generateContentSection($section);
    }
    
    $html .= '
    </main>
    
    <!-- 页脚 -->
    <footer class="bg-gray-100 border-t py-8 mt-12">
        <div class="container mx-auto px-4 text-center text-gray-600">
            <p>&copy; ' . date('Y') . ' ' . htmlspecialchars($data['title']) . '. 保留所有权利.</p>
        </div>
    </footer>
    
    <!-- 交互脚本 -->
    <script>
    document.addEventListener("DOMContentLoaded", function() {
        // 移动菜单切换
        var menuButton = document.getElementById("menu-button");
        var mobileMenu = document.getElementById("mobile-menu");
        
        if (menuButton && mobileMenu) {
            menuButton.addEventListener("click", function() {
                mobileMenu.classList.toggle("hidden");
            });
        }
        
        // 平滑滚动
        document.querySelectorAll(\'a[href^="#"]\').forEach(anchor => {
            anchor.addEventListener("click", function(e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute("href"));
                if (target) {
                    window.scrollTo({
                        top: target.offsetTop - 80,
                        behavior: "smooth"
                    });
                    mobileMenu.classList.add("hidden");
                }
            });
        });
    });
    </script>
</body>
</html>';
    
    return $html;
}

// 5. 内容区块生成函数
function generateContentSection($section) {
    $sectionId = 'section-' . preg_replace('/[^a-z0-9]+/', '-', strtolower($section['title']));
    
    $html = '
        <section id="' . $sectionId . '" class="mb-12 content-block bg-white rounded-xl shadow-sm p-6 md:p-8">
            <h' . $section['level'] . ' class="text-' . ($section['level'] + 1) . 'xl font-bold text-gray-800 mb-6 pb-2 border-b border-gray-200">' . 
                htmlspecialchars($section['title']) . 
            '</h' . $section['level'] . '>
            <div class="prose max-w-none text-gray-700 space-y-4">';
    
    foreach ($section['content'] as $content) {
        switch ($content['type']) {
            case 'paragraph':
                $html .= '
                <p>' . $content['text'] . '</p>';
                break;
                
            case 'list':
                $html .= '
                <ul class="list-disc pl-6 space-y-2">';
                foreach ($content['items'] as $item) {
                    $html .= '
                    <li>' . $item . '</li>';
                }
                $html .= '
                </ul>';
                break;
                
            default:
                $html .= '
                <p>' . htmlspecialchars(json_encode($content)) . '</p>';
        }
    }
    
    $html .= '
            </div>
        </section>';
    
    return $html;
}

// 6. PHP5.2兼容的JSON编码函数
if (!function_exists('json_encode')) {
    function custom_json_encode($data) {
        if (is_null($data)) return 'null';
        if ($data === false) return 'false';
        if ($data === true) return 'true';
        if (is_scalar($data)) {
            if (is_float($data)) {
                return floatval(str_replace(",", ".", strval($data)));
            }
            if (is_string($data)) {
                static $jsonReplaces = array(array("\\", "/", "\n", "\t", "\r", "\b", "\f", '"'), array('\\\\', '\\/', '\\n', '\\t', '\\r', '\\b', '\\f', '\"'));
                return '"' . str_replace($jsonReplaces[0], $jsonReplaces[1], $data) . '"';
            }
            return $data;
        }
        $isList = true;
        for ($i = 0, reset($data); $i < count($data); $i++, next($data)) {
            if (key($data) !== $i) {
                $isList = false;
                break;
            }
        }
        $result = array();
        if ($isList) {
            foreach ($data as $v) $result[] = custom_json_encode($v);
            return '[' . join(',', $result) . ']';
        } else {
            foreach ($data as $k => $v) $result[] = custom_json_encode($k).':'.custom_json_encode($v);
            return '{' . join(',', $result) . '}';
        }
    }
}

// 7. 生成并输出页面
header('Content-Type: text/html; charset=UTF-8');
echo generateHtmlPage($contentData);
?>