<?php
// api.php - 单文件数据库查询API
/*获取第一页内容列表（HTML）：api.php?action=list

获取第二页内容列表（JSON）：api.php?action=list&page=2&type=json

搜索包含"塑胶跑道"的内容：api.php?action=list&keyword=塑胶跑道

获取ID为5的内容详情（HTML）：api.php?action=view&id=5

获取ID为5的内容详情（JSON）：api.php?action=view&id=5&type=json
*/
// 配置
define('DB_FILE', 'content_ai.db');
define('ITEMS_PER_PAGE', 10);
// 允许所有域名跨域访问
header("Access-Control-Allow-Origin: *");
// 允许的HTTP方法
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
// 允许的请求头
header("Access-Control-Allow-Headers: Content-Type, Authorization");
// 初始化数据库连接
function initDB() {
    if (!file_exists(DB_FILE)) {
        die("Database file not found");
    }
    $db = new SQLite3(DB_FILE);
    return $db;
}

// 获取内容列表
function getContents($page = 1, $keyword = '', $categoryId = null) {
    $db = initDB();
    $offset = ($page - 1) * ITEMS_PER_PAGE;
    
    // 构建查询条件
    $where = [];
    $params = [];
    
    if (!empty($keyword)) {
        $where[] = "(gc.question LIKE :keyword OR gc.answer LIKE :keyword)";
        $params[':keyword'] = "%$keyword%";
    }
    
    if ($categoryId) {
        $where[] = "gc.category_id = :category_id";
        $params[':category_id'] = $categoryId;
    }
    
    $whereClause = $where ? "WHERE " . implode(" AND ", $where) : "";
    
    // 主查询
    $query = "
        SELECT gc.id, gc.question, gc.answer, gc.created_at, 
               c.id as category_id, c.name as category_name 
        FROM generated_content gc
        LEFT JOIN categories c ON gc.category_id = c.id
        $whereClause
        ORDER BY gc.created_at DESC 
        LIMIT " . ITEMS_PER_PAGE . " OFFSET $offset
    ";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    
    $result = $stmt->execute();
    $contents = [];
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $contents[] = $row;
    }
    
    // 获取总数
    $countQuery = "SELECT COUNT(*) as total FROM generated_content gc $whereClause";
    $stmt = $db->prepare($countQuery);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $total = $stmt->execute()->fetchArray(SQLITE3_ASSOC)['total'];
    
    // 获取分类列表
    $categories = [];
    $catResult = $db->query("SELECT id, name FROM categories ORDER BY name");
    while ($cat = $catResult->fetchArray(SQLITE3_ASSOC)) {
        $categories[] = $cat;
    }
    
    $db->close();
    
    return [
        'contents' => $contents,
        'total' => $total,
        'total_pages' => ceil($total / ITEMS_PER_PAGE),
        'current_page' => $page,
        'categories' => $categories,
        'current_category' => $categoryId,
        'keyword' => $keyword
    ];
}

// 获取单个内容详情
function getContent($id) {
    $db = initDB();
    $stmt = $db->prepare('
        SELECT gc.id, gc.question, gc.answer, gc.created_at, 
               c.id as category_id, c.name as category_name 
        FROM generated_content gc
        LEFT JOIN categories c ON gc.category_id = c.id
        WHERE gc.id = :id
    ');
    $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
    $result = $stmt->execute();
    $content = $result->fetchArray(SQLITE3_ASSOC);
    $db->close();
    
    return $content;
}

// 输出JSON响应
function jsonResponse($data) {
    header('Content-Type: application/json');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// 输出HTML列表页
function htmlListPage($data) {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>内容列表</title>
        <link href="https://cdn.bootcdn.net/ajax/libs/picocss/2.1.0/pico.css" rel="stylesheet">
        <style>
            .pagination { margin: 20px 0; }
            .pagination a, .pagination strong { 
                display: inline-block; 
                padding: 5px 10px; 
                margin: 0 2px; 
                border: 1px solid #ddd; 
                text-decoration: none;
            }
            .pagination a:hover { background: #eee; }
            .pagination strong { background: #0073aa; color: white; }
            .search-form { margin-bottom: 20px; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>内容列表</h1>
            
            <form class="search-form" method="get">
                <input type="hidden" name="action" value="list">
                <div class="grid">
                    <div>
                        <input type="text" name="keyword" placeholder="搜索关键词" value="<?= htmlspecialchars($data['keyword']) ?>">
                    </div>
                    <div>
                        <select name="category">
                            <option value="">所有分类</option>
                            <?php foreach ($data['categories'] as $cat): ?>
                                <option value="<?= $cat['id'] ?>" <?= $data['current_category'] == $cat['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($cat['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <button type="submit">搜索</button>
                    </div>
                </div>
            </form>
            
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>分类</th>
                        <th>问题</th>
                        <th>创建时间</th>
                        <th>操作</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($data['contents'] as $item): ?>
                        <tr>
                            <td><?= htmlspecialchars($item['id']) ?></td>
                            <td><?= htmlspecialchars($item['category_name']) ?></td>
                            <td><?= htmlspecialchars(mb_substr($item['question'], 0, 50, 'UTF-8')) ?></td>
                            <td><?= htmlspecialchars($item['created_at']) ?></td>
                            <td>
                                <a href="?action=view&id=<?= $item['id'] ?>">查看</a>
                                <a href="?action=view&id=<?= $item['id'] ?>&type=json">JSON</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <?php if ($data['total_pages'] > 1): ?>
                <div class="pagination">
                    <?php if ($data['current_page'] > 1): ?>
                        <a href="?action=list&page=1&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">首页</a>
                        <a href="?action=list&page=<?= $data['current_page']-1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">上一页</a>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $data['total_pages']; $i++): ?>
                        <?php if ($i == $data['current_page']): ?>
                            <strong><?= $i ?></strong>
                        <?php else: ?>
                            <a href="?action=list&page=<?= $i ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>"><?= $i ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <?php if ($data['current_page'] < $data['total_pages']): ?>
                        <a href="?action=list&page=<?= $data['current_page']+1 ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">下一页</a>
                        <a href="?action=list&page=<?= $data['total_pages'] ?>&keyword=<?= urlencode($data['keyword']) ?>&category=<?= $data['current_category'] ?>">末页</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </body>
    </html>
    <?php
}

// 输出HTML详情页
function htmlViewPage($content) {
    if (!$content) {
        echo "<p>内容不存在</p>";
        echo "<a href='?action=list'>返回列表</a>";
        return;
    }
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title><?= htmlspecialchars(mb_substr($content['question'], 0, 20, 'UTF-8')) ?></title>
        <link href="https://cdn.bootcdn.net/ajax/libs/picocss/2.1.0/pico.css" rel="stylesheet">
    </head>
    <body>
        <div class="container">
            <h1><?= htmlspecialchars($content['question']) ?></h1>
            <p>分类: <?= htmlspecialchars($content['category_name']) ?></p>
            <p>创建时间: <?= htmlspecialchars($content['created_at']) ?></p>
            
            <div style="border:1px solid #ccc; padding:15px; margin:15px 0;">
                <?= nl2br(htmlspecialchars($content['answer'])) ?>
            </div>
            
            <div>
                <a href="?action=list">返回列表</a> | 
                <a href="?action=view&id=<?= $content['id'] ?>&type=json">查看JSON格式</a>
            </div>
        </div>
    </body>
    </html>
    <?php
}

// 主控制器
function handleRequest() {
    $action = $_GET['action'] ?? 'list';
    $outputType = $_GET['type'] ?? 'html';
    
    try {
        switch ($action) {
            case 'list':
                $page = max(1, intval($_GET['page'] ?? 1));
                $keyword = trim($_GET['keyword'] ?? '');
                $categoryId = @$_GET['category'] ? intval(@$_GET['category']) : null;
                
                $data = getContents($page, $keyword, $categoryId);
                
                if ($outputType === 'json') {
                    jsonResponse($data);
                } else {
                    htmlListPage($data);
                }
                break;
                
            case 'view':
                if (!isset($_GET['id'])) {
                    throw new Exception("缺少ID参数");
                }
                
                $id = intval($_GET['id']);
                $content = getContent($id);
                
                if ($outputType === 'json') {
                    jsonResponse($content ?: ['error' => 'Content not found']);
                } else {
                    htmlViewPage($content);
                }
                break;
                
            default:
                throw new Exception("无效的操作");
        }
    } catch (Exception $e) {
        if ($outputType === 'json') {
            jsonResponse(['error' => $e->getMessage()]);
        } else {
            echo "<p>错误: " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    }
}

// 启动应用
handleRequest();
?>