<?php
class TextReplacer {
    private $replacements = [];
    private $regexReplacements = [];
    private $dictionaryPath = '';
    
    /**
     * Constructor
     * @param string $dictionaryPath Path to dictionary file
     */
    public function __construct($dictionaryPath = '') {
        if ($dictionaryPath) {
            $this->setDictionaryPath($dictionaryPath);
        }
        
        // Preset some common rules
        $this->presetCommonRules();
    }
    
    /**
     * Set dictionary file path
     * @param string $path Path to dictionary file
     * @throws Exception If file not found
     */
    public function setDictionaryPath($path) {
        if (file_exists($path)) {
            $this->dictionaryPath = $path;
            $this->loadDictionary();
        } else {
            throw new Exception("Dictionary file not found: " . $path);
        }
    }
    
    /**
     * Load dictionary file
     */
    private function loadDictionary() {
        $lines = file($this->dictionaryPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
            // Skip comments
            if (strpos(trim($line), '#') === 0) {
                continue;
            }
            
            // Parse replacement rule
            $parts = explode('=>', $line, 2);
            if (count($parts) === 2) {
                $search = trim($parts[0]);
                $replace = trim($parts[1]);
                
                // Check if it's a regex pattern (starts and ends with /)
                if (preg_match('/^\/.+\/[a-zA-Z]*$/u', $search)) {
                    $this->addRegexReplacement($search, $replace);
                } else {
                    $this->addReplacement($search, $replace);
                }
            }
        }
    }
    
    /**
     * Add simple replacement rule
     * @param string $search  String to search for
     * @param string $replace Replacement string
     */
    public function addReplacement($search, $replace) {
        $this->replacements[$search] = $replace;
    }
    
    /**
     * Add regex replacement rule
     * @param string $pattern Regex pattern
     * @param string $replace Replacement string or callback
     */
    public function addRegexReplacement($pattern, $replace) {
        $this->regexReplacements[$pattern] = $replace;
    }
    
    /**
     * Add multiple replacement rules
     * @param array $rules Array of replacement rules
     */
    public function addReplacements(array $rules) {
        foreach ($rules as $search => $replace) {
            if (preg_match('/^\/.+\/[a-zA-Z]*$/u', $search)) {
                $this->addRegexReplacement($search, $replace);
            } else {
                $this->addReplacement($search, $replace);
            }
        }
    }
    
    /**
     * Preset common replacement rules
     */
    private function presetCommonRules() {
        // Common spelling corrections
        $this->addReplacement('teh', 'the');
        $this->addReplacement('adn', 'and');
        $this->addReplacement('thier', 'their');
        
        // Common symbol replacements
        $this->addReplacement('&', 'and');
        $this->addReplacement('@', 'at');
        
        // Common ordinal number replacements
        $this->addRegexReplacement('/\b(\d+)st\b/u', '$1th');
        $this->addRegexReplacement('/\b(\d+)nd\b/u', '$1th');
        $this->addRegexReplacement('/\b(\d+)rd\b/u', '$1th');
        $this->addRegexReplacement('/\b(\d+)th\b/u', '$1th');
    }
    
    /**
     * Perform all replacements on text
     * @param string $text Input text
     * @return string Processed text
     */
    public function replace($text) {
        // Ensure UTF-8 encoding
        if (!mb_check_encoding($text, 'UTF-8')) {
            $text = mb_convert_encoding($text, 'UTF-8');
        }
        
        // First do simple replacements
        if (!empty($this->replacements)) {
            $text = str_replace(
                array_keys($this->replacements),
                array_values($this->replacements),
                $text
            );
        }
        
        // Then do regex replacements
        if (!empty($this->regexReplacements)) {
            foreach ($this->regexReplacements as $pattern => $replacement) {
                // Handle callback functions
                if (is_callable($replacement)) {
                    $text = preg_replace_callback($pattern, $replacement, $text);
                } 
                // Handle string replacements with possible callbacks
                elseif (is_string($replacement) && preg_match('/\b(strtolower|strtoupper|ucfirst|ucwords)\s*\(/', $replacement)) {
                    $text = preg_replace_callback($pattern, 
                        function($matches) use ($replacement) {
                            return eval('return ' . $replacement . ';');
                        }, 
                        $text);
                }
                // Standard regex replacement
                else {
                    $text = preg_replace($pattern, $replacement, $text);
                }
            }
        }
        
        return $text;
    }
    
    /**
     * Get all current replacement rules
     * @return array Array of all replacement rules
     */
    public function getRules() {
        return [
            'simple' => $this->replacements,
            'regex' => $this->regexReplacements
        ];
    }
    
    /**
     * Clear all replacement rules
     */
    public function clearRules() {
        $this->replacements = [];
        $this->regexReplacements = [];
    }
    
    /**
     * Check if a string is valid UTF-8
     * @param string $string String to check
     * @return bool True if valid UTF-8
     */
    private function isUtf8($string) {
        return mb_check_encoding($string, 'UTF-8');
    }
}