<?php
/**
 * Simple Blog System using RemoteDB API
 * 
 * Features:
 * - Unlimited category hierarchy
 * - Article management (CRUD)
 * - Tag management
 * - Custom slugs for all content
 * - Toggle pretty URLs
 * - Blog settings (site name, keywords, description, etc.)
 */

// Initialize RemoteDB
require (dirname(__file__) . '/RemoteDB.php');
$db = new RemoteDB('./blog.d3', 'https://atm-tkd.sakura.ne.jp/pcspe/access/');

// Initialize Blog System https://atm-tkd.sakura.ne.jp/pcspe/access/dbs.php?ac=create&db=./blog.d3
$blog = new BlogSystem($db);

// Handle requests
$blog->handleRequest();

class BlogSystem {
    private $db;
    private $settings;
    private $prettyUrls = false;
    
    public function __construct($db) {
        $this->db = $db;
        $this->initializeDatabase();
        $this->loadSettings();
        $this->detectPrettyUrls();
    }
    
    /**
     * Initialize database tables if they don't exist
     */
    private function initializeDatabase() {
        // Create settings table
        $this->db->exec("CREATE TABLE IF NOT EXISTS settings (
            id INTEGER PRIMARY KEY,
            name TEXT,
            value TEXT
        )");
        
        // Create categories table
        $this->db->exec("CREATE TABLE IF NOT EXISTS categories (
            id INTEGER PRIMARY KEY,
            parent_id INTEGER DEFAULT 0,
            name TEXT,
            slug TEXT UNIQUE,
            description TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create tags table
        $this->db->exec("CREATE TABLE IF NOT EXISTS tags (
            id INTEGER PRIMARY KEY,
            name TEXT,
            slug TEXT UNIQUE,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )");
        
        // Create posts table
        $this->db->exec("CREATE TABLE IF NOT EXISTS posts (
            id INTEGER PRIMARY KEY,
            title TEXT,
            slug TEXT UNIQUE,
            content TEXT,
            excerpt TEXT,
            category_id INTEGER,
            status TEXT DEFAULT 'publish',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            published_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY(category_id) REFERENCES categories(id)
        )");
        
        // Create post_tags junction table
        $this->db->exec("CREATE TABLE IF NOT EXISTS post_tags (
            post_id INTEGER,
            tag_id INTEGER,
            PRIMARY KEY(post_id, tag_id),
            FOREIGN KEY(post_id) REFERENCES posts(id),
            FOREIGN KEY(tag_id) REFERENCES tags(id)
        )");
    }
    
    /**
     * Load blog settings from database
     */
    private function loadSettings() {
        $results = $this->db->query("SELECT name, value FROM settings");
        #print_r($results);
        $this->settings = [];
        
        if ($results['data']) {
            foreach ($results['data'] as $row) {
            #print_r($row);
                $this->settings[$row['name']] = $row['value'];
            }
        }
        
        // Set defaults if not exists
        $defaults = [
            'site_title' => 'My Blog',
            'site_description' => 'A simple blog system',
            'site_keywords' => 'blog, simple, php',
            'footer_text' => 'Powered by Simple Blog',
            'icp_number' => '',
            'pretty_urls' => '0'
        ];
        
        foreach ($defaults as $key => $value) {
            if (!isset($this->settings[$key])) {
                $this->settings[$key] = $value;
                $this->db->exec("INSERT INTO settings(name, value) VALUES(?, ?)", [$key, $value]);
            }
        }
    }
    
    /**
     * Detect if pretty URLs are enabled
     */
    private function detectPrettyUrls() {
        $this->prettyUrls = ($this->settings['pretty_urls'] === '1');
    }
    
    /**
     * Handle incoming requests and route them
     */
    public function handleRequest() {
        $action = $_GET['action'] ?? '';
        $id = $_GET['id'] ?? 0;
        
        // Admin actions
        if (isset($_GET['admin'])) {
            $this->checkAdminAuth();
            $this->handleAdminRequest($action, $id);
            return;
        }
        
        // Frontend actions
        switch ($action) {
            case 'post':
                $this->showPost($id);
                break;
            case 'category':
                $this->showCategory($id);
                break;
            case 'tag':
                $this->showTag($id);
                break;
            default:
                $this->showHomepage();
        }
    }
    
    /**
     * Handle admin requests
     */
    private function handleAdminRequest($action, $id) {
        switch ($action) {
            case 'save_settings':
                $this->saveSettings();
                break;
            case 'toggle_pretty_urls':
                $this->togglePrettyUrls();
                break;
            case 'edit_category':
                $this->editCategory($id);
                break;
            case 'save_category':
                $this->saveCategory();
                break;
            case 'delete_category':
                $this->deleteCategory($id);
                break;
            case 'edit_post':
                $this->editPost($id);
                break;
            case 'save_post':
                $this->savePost();
                break;
            case 'delete_post':
                $this->deletePost($id);
                break;
            case 'edit_tag':
                $this->editTag($id);
                break;
            case 'save_tag':
                $this->saveTag();
                break;
            case 'delete_tag':
                $this->deleteTag($id);
                break;
            default:
                $this->showAdminDashboard();
        }
    }
    
    /**
     * Check admin authentication
     */
    private function checkAdminAuth() {
        @session_start();
        
        // Simple authentication - in a real app, use proper authentication
        if (!isset($_SESSION['admin_logged_in'])) {
            if (@$_POST['username'] === 'admin' && @$_POST['password'] === 'asasas888') {
                $_SESSION['admin_logged_in'] = true;
            } else {
                $this->showAdminLogin();
                exit;
            }
        }
    }
    
    /**
     * Show admin login form
     */
    private function showAdminLogin() {
        echo $this->renderTemplate('admin/login', [
            'site_title' => $this->settings['site_title']
        ]);
    }
    
    /**
     * Show admin dashboard
     */
    private function showAdminDashboard() {
        $posts = $this->db->query("SELECT * FROM posts ORDER BY created_at DESC LIMIT 5");
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        $tags = $this->db->query("SELECT * FROM tags ORDER BY name");
        
        echo $this->renderTemplate('admin/dashboard', [
            'settings' => $this->settings,
            'posts' => $posts,
            'categories' => $categories,
            'tags' => $tags,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    /**
     * Save blog settings
     */
    private function saveSettings() {
        foreach ($_POST as $name => $value) {
            if (array_key_exists($name, $this->settings)) {
                $this->db->exec("UPDATE settings SET value = ? WHERE name = ?", [$value, $name]);
                $this->settings[$name] = $value;
            }
        }
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Toggle pretty URLs setting
     */
    private function togglePrettyUrls() {
        $newValue = $this->prettyUrls ? '0' : '1';
        $this->db->exec("UPDATE settings SET value = ? WHERE name = 'pretty_urls'", [$newValue]);
        $this->settings['pretty_urls'] = $newValue;
        $this->prettyUrls = !$this->prettyUrls;
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Show category edit form
     */
    private function editCategory($id = 0) {
        $category = $id ? $this->db->query("SELECT * FROM categories WHERE id = ?", [$id])[0] : null;
        $categories = $this->db->query("SELECT * FROM categories WHERE id != ? ORDER BY name", [$id]);
        
        echo $this->renderTemplate('admin/edit_category', [
            'category' => $category,
            'categories' => $categories,
            'settings' => $this->settings
        ]);
    }
    
    /**
     * Save category
     */
    private function saveCategory() {
        $id = $_POST['id'] ?? 0;
        $name = $_POST['name'];
        $slug = $_POST['slug'] ?: $this->generateSlug($name);
        $parentId = $_POST['parent_id'] ?? 0;
        $description = $_POST['description'];
        
        if ($id) {
            $this->db->exec(
                "UPDATE categories SET name = ?, slug = ?, parent_id = ?, description = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
                [$name, $slug, $parentId, $description, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO categories(name, slug, parent_id, description) VALUES(?, ?, ?, ?)",
                [$name, $slug, $parentId, $description]
            );
        }
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Delete category
     */
    private function deleteCategory($id) {
        // Move posts to default category
        $this->db->exec("UPDATE posts SET category_id = 0 WHERE category_id = ?", [$id]);
        // Delete category
        $this->db->exec("DELETE FROM categories WHERE id = ?", [$id]);
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Show post edit form
     */
    private function editPost($id = 0) {
        $post = $id ? $this->db->query("SELECT * FROM posts WHERE id = ?", [$id])[0] : null;
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        $tags = $this->db->query("SELECT * FROM tags ORDER BY name");
        $postTags = $id ? $this->db->query("SELECT tag_id FROM post_tags WHERE post_id = ?", [$id]) : [];
        $postTagIds = array_column($postTags, 'tag_id');
        
        echo $this->renderTemplate('admin/edit_post', [
            'post' => $post,
            'categories' => $categories,
            'tags' => $tags,
            'postTagIds' => $postTagIds,
            'settings' => $this->settings
        ]);
    }
    
    /**
     * Save post
     */
    private function savePost() {
        $id = $_POST['id'] ?? 0;
        $title = $_POST['title'];
        $slug = $_POST['slug'] ?: $this->generateSlug($title);
        $content = $_POST['content'];
        $excerpt = $_POST['excerpt'] ?: substr(strip_tags($content), 0, 200);
        $categoryId = $_POST['category_id'] ?? 0;
        $status = $_POST['status'] ?? 'publish';
        $tags = $_POST['tags'] ?? [];
        
        if ($id) {
            $this->db->exec(
                "UPDATE posts SET title = ?, slug = ?, content = ?, excerpt = ?, category_id = ?, status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?",
                [$title, $slug, $content, $excerpt, $categoryId, $status, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO posts(title, slug, content, excerpt, category_id, status) VALUES(?, ?, ?, ?, ?, ?)",
                [$title, $slug, $content, $excerpt, $categoryId, $status]
            );
            $id = $this->db->query("SELECT last_insert_rowid() as id")['data'][0]['id'];
        }
        
        // Update tags
        $this->db->exec("DELETE FROM post_tags WHERE post_id = ?", [$id]);
        foreach ($tags as $tagId) {
            $this->db->exec("INSERT INTO post_tags(post_id, tag_id) VALUES(?, ?)", [$id, $tagId]);
        }
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Delete post
     */
    private function deletePost($id) {
        $this->db->exec("DELETE FROM posts WHERE id = ?", [$id]);
        $this->db->exec("DELETE FROM post_tags WHERE post_id = ?", [$id]);
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Show tag edit form
     */
    private function editTag($id = 0) {
        $tag = $id ? $this->db->query("SELECT * FROM tags WHERE id = ?", [$id])[0] : null;
        
        echo $this->renderTemplate('admin/edit_tag', [
            'tag' => $tag,
            'settings' => $this->settings
        ]);
    }
    
    /**
     * Save tag
     */
    private function saveTag() {
        $id = $_POST['id'] ?? 0;
        $name = $_POST['name'];
        $slug = $_POST['slug'] ?: $this->generateSlug($name);
        
        if ($id) {
            $this->db->exec(
                "UPDATE tags SET name = ?, slug = ? WHERE id = ?",
                [$name, $slug, $id]
            );
        } else {
            $this->db->exec(
                "INSERT INTO tags(name, slug) VALUES(?, ?)",
                [$name, $slug]
            );
        }
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Delete tag
     */
    private function deleteTag($id) {
        $this->db->exec("DELETE FROM tags WHERE id = ?", [$id]);
        $this->db->exec("DELETE FROM post_tags WHERE tag_id = ?", [$id]);
        
        header("Location: ?admin&action=dashboard");
    }
    
    /**
     * Show homepage
     */
    private function showHomepage() {
        $posts = $this->db->query("SELECT * FROM posts WHERE status = 'publish' ORDER BY published_at DESC LIMIT 10");
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('home', [
            'posts' => $posts,
            'categories' => $categories,
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    /**
     * Show single post
     */
    private function showPost($idOrSlug) {
        #echo $idOrSlug;
        if (is_numeric($idOrSlug)) {
            $post = $this->db->query("SELECT * FROM posts WHERE id = ?", [$idOrSlug])['data']['0'];
        } else {
            $post = $this->db->query("SELECT * FROM posts WHERE slug = ?", [$idOrSlug])['data']['0'];
        }
        print_r($post);
        if (!$post) {
            $this->show404();
            return;
        }
        
        $category = $post['category_id'] ? $this->db->query("SELECT * FROM categories WHERE id = ?", [$post['category_id']])['data'][0] : null;
        $tags = $this->db->query("SELECT t.* FROM tags t JOIN post_tags pt ON t.id = pt.tag_id WHERE pt.post_id = ?", [$post['id']])['data'][0];
        
        echo $this->renderTemplate('post', [
            'post' => $post,
            'category' => $category,
            'tags' => $tags,
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    /**
     * Show category archive
     */
    private function showCategory($idOrSlug) {
        if (is_numeric($idOrSlug)) {
            $category = $this->db->query("SELECT * FROM categories WHERE id = ?", [$idOrSlug])[0];
        } else {
            $category = $this->db->query("SELECT * FROM categories WHERE slug = ?", [$idOrSlug])[0];
        }
        
        if (!$category) {
            $this->show404();
            return;
        }
        
        $posts = $this->db->query("SELECT * FROM posts WHERE category_id = ? AND status = 'publish' ORDER BY published_at DESC", [$category['id']]);
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('category', [
            'category' => $category,
            'posts' => $posts,
            'categories' => $categories,
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    /**
     * Show tag archive
     */
    private function showTag($idOrSlug) {
        if (is_numeric($idOrSlug)) {
            $tag = $this->db->query("SELECT * FROM tags WHERE id = ?", [$idOrSlug])[0];
        } else {
            $tag = $this->db->query("SELECT * FROM tags WHERE slug = ?", [$idOrSlug])[0];
        }
        
        if (!$tag) {
            $this->show404();
            return;
        }
        
        $posts = $this->db->query("
            SELECT p.* 
            FROM posts p 
            JOIN post_tags pt ON p.id = pt.post_id 
            WHERE pt.tag_id = ? AND p.status = 'publish' 
            ORDER BY p.published_at DESC
        ", [$tag['id']]);
        
        $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
        
        echo $this->renderTemplate('tag', [
            'tag' => $tag,
            'posts' => $posts,
            'categories' => $categories,
            'settings' => $this->settings,
            'prettyUrls' => $this->prettyUrls
        ]);
    }
    
    /**
     * Show 404 page
     */
    private function show404() {
        header("HTTP/1.0 404 Not Found");
        echo $this->renderTemplate('404', [
            'settings' => $this->settings
        ]);
    }
    
    /**
     * Generate URL slug from text
     */
    private function generateSlug($text) {
        $slug = preg_replace('/[^a-z0-9]+/', '-', strtolower($text));
        $slug = trim($slug, '-');
        return $slug;
    }
    
    /**
     * Render template with variables
     */
    private function renderTemplate($template, $vars = []) {
        ob_start();
        extract($vars);
        include __DIR__ . "/templates/$template.php";
        return ob_get_clean();
    }
}

// Simple template files would be in a templates/ directory
// For this single-file example, we'll define them as strings
if (!function_exists('renderTemplate')) {
    function renderTemplate($template, $vars = []) {
        extract($vars);
        ob_start();
        
        switch ($template) {
            case 'admin/login':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Login - <?= htmlspecialchars($site_title) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 400px; margin: 0 auto; padding: 20px; }
                        input { width: 100%; padding: 8px; margin: 8px 0; box-sizing: border-box; }
                        button { background: #333; color: white; padding: 10px; border: none; width: 100%; }
                    </style>
                </head>
                <body>
                    <h1>Admin Login</h1>
                    <form method="post">
                        <input type="text" name="username" placeholder="Username" required>
                        <input type="password" name="password" placeholder="Password" required>
                        <button type="submit">Login</button>
                    </form>
                </body>
                </html>
                <?php
                break;
                
            case 'admin/dashboard':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Dashboard - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
                        .header { background: #333; color: white; padding: 10px 20px; }
                        .sidebar { float: left; width: 200px; background: #f4f4f4; height: 100vh; padding: 20px; }
                        .content { margin-left: 240px; padding: 20px; }
                        table { width: 100%; border-collapse: collapse; }
                        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                        th { background: #f4f4f4; }
                        .btn { display: inline-block; padding: 6px 12px; background: #333; color: white; text-decoration: none; }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1><?= htmlspecialchars($settings['site_title']) ?> Admin</h1>
                    </div>
                    <div class="sidebar">
                        <h3>Menu</h3>
                        <ul>
                            <li><a href="?admin">Dashboard</a></li>
                            <li><a href="?admin&action=edit_post">Add New Post</a></li>
                            <li><a href="?admin&action=edit_category">Add New Category</a></li>
                            <li><a href="?admin&action=edit_tag">Add New Tag</a></li>
                            <li><a href="?admin&action=save_settings">Settings</a></li>
                            <li><a href="?admin&action=toggle_pretty_urls">Toggle Pretty URLs (<?= $prettyUrls ? 'ON' : 'OFF' ?>)</a></li>
                            <li><a href="?" target="_blank">View Site</a></li>
                            <li><a href="?admin&logout=1">Logout</a></li>
                        </ul>
                    </div>
                    <div class="content">
                        <h2>Recent Posts</h2>
                        <table>
                            <tr>
                                <th>Title</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                            <?php foreach ($posts as $post): ?>
                            <tr>
                                <td><?= htmlspecialchars($post['title']) ?></td>
                                <td><?= date('Y-m-d', strtotime($post['created_at'])) ?></td>
                                <td>
                                    <a href="?admin&action=edit_post&id=<?= $post['id'] ?>" class="btn">Edit</a>
                                    <a href="?admin&action=delete_post&id=<?= $post['id'] ?>" class="btn" onclick="return confirm('Are you sure?')">Delete</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </table>
                        
                        <h2>Categories</h2>
                        <table>
                            <tr>
                                <th>Name</th>
                                <th>Actions</th>
                            </tr>
                            <?php foreach ($categories as $category): ?>
                            <tr>
                                <td><?= htmlspecialchars($category['name']) ?></td>
                                <td>
                                    <a href="?admin&action=edit_category&id=<?= $category['id'] ?>" class="btn">Edit</a>
                                    <a href="?admin&action=delete_category&id=<?= $category['id'] ?>" class="btn" onclick="return confirm('Are you sure?')">Delete</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </table>
                        
                        <h2>Tags</h2>
                        <table>
                            <tr>
                                <th>Name</th>
                                <th>Actions</th>
                            </tr>
                            <?php foreach ($tags as $tag): ?>
                            <tr>
                                <td><?= htmlspecialchars($tag['name']) ?></td>
                                <td>
                                    <a href="?admin&action=edit_tag&id=<?= $tag['id'] ?>" class="btn">Edit</a>
                                    <a href="?admin&action=delete_tag&id=<?= $tag['id'] ?>" class="btn" onclick="return confirm('Are you sure?')">Delete</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </table>
                    </div>
                </body>
                </html>
                <?php
                break;
                
            case 'admin/edit_category':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= $category ? 'Edit' : 'Add' ?> Category - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        input, textarea, select { width: 100%; padding: 8px; margin: 8px 0; box-sizing: border-box; }
                        button { background: #333; color: white; padding: 10px; border: none; width: 100%; }
                    </style>
                </head>
                <body>
                    <h1><?= $category ? 'Edit' : 'Add' ?> Category</h1>
                    <form method="post" action="?admin&action=save_category">
                        <input type="hidden" name="id" value="<?= $category['id'] ?? 0 ?>">
                        
                        <label for="name">Name:</label>
                        <input type="text" id="name" name="name" value="<?= htmlspecialchars($category['name'] ?? '') ?>" required>
                        
                        <label for="slug">Slug:</label>
                        <input type="text" id="slug" name="slug" value="<?= htmlspecialchars($category['slug'] ?? '') ?>">
                        
                        <label for="parent_id">Parent Category:</label>
                        <select id="parent_id" name="parent_id">
                            <option value="0">None</option>
                            <?php foreach ($categories as $cat): ?>
                                <?php if (isset($category['id']) && $cat['id'] == $category['id']) continue; ?>
                                <option value="<?= $cat['id'] ?>" <?= (isset($category['parent_id']) && $category['parent_id'] == $cat['id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($cat['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        
                        <label for="description">Description:</label>
                        <textarea id="description" name="description" rows="4"><?= htmlspecialchars($category['description'] ?? '') ?></textarea>
                        
                        <button type="submit">Save Category</button>
                    </form>
                    <p><a href="?admin">← Back to Dashboard</a></p>
                </body>
                </html>
                <?php
                break;
                
            case 'admin/edit_post':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= $post ? 'Edit' : 'Add' ?> Post - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        input, textarea, select { width: 100%; padding: 8px; margin: 8px 0; box-sizing: border-box; }
                        button { background: #333; color: white; padding: 10px; border: none; width: 100%; }
                        .tag-list { margin: 10px 0; }
                        .tag-list label { display: inline-block; margin-right: 10px; }
                    </style>
                </head>
                <body>
                    <h1><?= $post ? 'Edit' : 'Add' ?> Post</h1>
                    <form method="post" action="?admin&action=save_post">
                        <input type="hidden" name="id" value="<?= $post['id'] ?? 0 ?>">
                        
                        <label for="title">Title:</label>
                        <input type="text" id="title" name="title" value="<?= htmlspecialchars($post['title'] ?? '') ?>" required>
                        
                        <label for="slug">Slug:</label>
                        <input type="text" id="slug" name="slug" value="<?= htmlspecialchars($post['slug'] ?? '') ?>">
                        
                        <label for="category_id">Category:</label>
                        <select id="category_id" name="category_id">
                            <option value="0">Uncategorized</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?= $cat['id'] ?>" <?= (isset($post['category_id']) && $post['category_id'] == $cat['id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($cat['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        
                        <label for="status">Status:</label>
                        <select id="status" name="status">
                            <option value="publish" <?= (isset($post['status']) && $post['status'] == 'publish') ? 'selected' : '' ?>>Published</option>
                            <option value="draft" <?= (isset($post['status']) && $post['status'] == 'draft') ? 'selected' : '' ?>>Draft</option>
                        </select>
                        
                        <label for="content">Content:</label>
                        <textarea id="content" name="content" rows="10" required><?= htmlspecialchars($post['content'] ?? '') ?></textarea>
                        
                        <label for="excerpt">Excerpt:</label>
                        <textarea id="excerpt" name="excerpt" rows="4"><?= htmlspecialchars($post['excerpt'] ?? '') ?></textarea>
                        
                        <div class="tag-list">
                            <strong>Tags:</strong><br>
                            <?php foreach ($tags as $tag): ?>
                                <label>
                                    <input type="checkbox" name="tags[]" value="<?= $tag['id'] ?>" <?= in_array($tag['id'], $postTagIds ?? []) ? 'checked' : '' ?>>
                                    <?= htmlspecialchars($tag['name']) ?>
                                </label>
                            <?php endforeach; ?>
                        </div>
                        
                        <button type="submit">Save Post</button>
                    </form>
                    <p><a href="?admin">← Back to Dashboard</a></p>
                </body>
                </html>
                <?php
                break;
                
            case 'admin/edit_tag':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= $tag ? 'Edit' : 'Add' ?> Tag - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        input { width: 100%; padding: 8px; margin: 8px 0; box-sizing: border-box; }
                        button { background: #333; color: white; padding: 10px; border: none; width: 100%; }
                    </style>
                </head>
                <body>
                    <h1><?= $tag ? 'Edit' : 'Add' ?> Tag</h1>
                    <form method="post" action="?admin&action=save_tag">
                        <input type="hidden" name="id" value="<?= $tag['id'] ?? 0 ?>">
                        
                        <label for="name">Name:</label>
                        <input type="text" id="name" name="name" value="<?= htmlspecialchars($tag['name'] ?? '') ?>" required>
                        
                        <label for="slug">Slug:</label>
                        <input type="text" id="slug" name="slug" value="<?= htmlspecialchars($tag['slug'] ?? '') ?>">
                        
                        <button type="submit">Save Tag</button>
                    </form>
                    <p><a href="?admin">← Back to Dashboard</a></p>
                </body>
                </html>
                <?php
                break;
                
            case 'home':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= htmlspecialchars($settings['site_title']) ?></title>
                    <meta name="description" content="<?= htmlspecialchars($settings['site_description']) ?>">
                    <meta name="keywords" content="<?= htmlspecialchars($settings['site_keywords']) ?>">
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        .post { margin-bottom: 40px; }
                        .post-title { font-size: 24px; margin-bottom: 5px; }
                        .post-meta { color: #666; font-size: 14px; margin-bottom: 10px; }
                        .post-excerpt { margin-bottom: 10px; }
                        .read-more { color: #333; font-weight: bold; }
                        .sidebar { float: right; width: 200px; margin-left: 20px; }
                        .category-list, .tag-list { margin-bottom: 20px; }
                        .footer { margin-top: 40px; border-top: 1px solid #eee; padding-top: 20px; }
                    </style>
                </head>
                <body>
                    <header>
                        <h1><?= htmlspecialchars($settings['site_title']) ?></h1>
                        <p><?= htmlspecialchars($settings['site_description']) ?></p>
                        <p><a href="?admin">Admin</a></p>
                    </header>
                    
                    <div class="sidebar">
                        <h3>Categories</h3>
                        <ul class="category-list">
                            <?php foreach ($categories as $category): ?>
                                <li>
                                    <a href="<?= $prettyUrls ? '/category/' . $category['slug'] : '?action=category&id=' . $category['id'] ?>">
                                        <?= htmlspecialchars($category['name']) ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div class="content">
                        <?php foreach ($posts as $post): ?>
                            <article class="post">
                                <h2 class="post-title">
                                    <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>">
                                        <?= htmlspecialchars($post['title']) ?>
                                    </a>
                                </h2>
                                <div class="post-meta">
                                    Posted on <?= date('F j, Y', strtotime($post['published_at'])) ?>
                                </div>
                                <div class="post-excerpt">
                                    <?= htmlspecialchars($post['excerpt']) ?>
                                </div>
                                <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>" class="read-more">Read more →</a>
                            </article>
                        <?php endforeach; ?>
                    </div>
                    
                    <footer class="footer">
                        <p><?= htmlspecialchars($settings['footer_text']) ?></p>
                        <?php if ($settings['icp_number']): ?>
                            <p>ICP: <?= htmlspecialchars($settings['icp_number']) ?></p>
                        <?php endif; ?>
                    </footer>
                </body>
                </html>
                <?php
                break;
                
            case 'post':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= htmlspecialchars($post['title']) ?> - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <meta name="description" content="<?= htmlspecialchars($post['excerpt']) ?>">
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        .post-title { font-size: 28px; margin-bottom: 10px; }
                        .post-meta { color: #666; font-size: 14px; margin-bottom: 20px; }
                        .post-content { line-height: 1.6; }
                        .tags { margin-top: 30px; }
                        .tag { display: inline-block; background: #f4f4f4; padding: 3px 8px; margin-right: 5px; }
                        .sidebar { float: right; width: 200px; margin-left: 20px; }
                        .category-list, .tag-list { margin-bottom: 20px; }
                        .footer { margin-top: 40px; border-top: 1px solid #eee; padding-top: 20px; }
                    </style>
                </head>
                <body>
                    <header>
                        <h1><a href="<?= $prettyUrls ? '/' : '?' ?>"><?= htmlspecialchars($settings['site_title']) ?></a></h1>
                        <p><a href="?admin">Admin</a></p>
                    </header>
                    
                    <div class="sidebar">
                        <h3>Categories</h3>
                        <ul class="category-list">
                            <?php 
                            $categories = $this->db->query("SELECT * FROM categories ORDER BY name");
                            foreach ($categories as $cat): ?>
                                <li>
                                    <a href="<?= $prettyUrls ? '/category/' . $cat['slug'] : '?action=category&id=' . $cat['id'] ?>">
                                        <?= htmlspecialchars($cat['name']) ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div class="content">
                        <article>
                            <h1 class="post-title"><?= htmlspecialchars($post['title']) ?></h1>
                            <div class="post-meta">
                                Posted on <?= date('F j, Y', strtotime($post['published_at'])) ?>
                                <?php if ($category): ?>
                                    in <a href="<?= $prettyUrls ? '/category/' . $category['slug'] : '?action=category&id=' . $category['id'] ?>">
                                        <?= htmlspecialchars($category['name']) ?>
                                    </a>
                                <?php endif; ?>
                            </div>
                            <div class="post-content">
                                <?= nl2br(htmlspecialchars($post['content'])) ?>
                            </div>
                            <?php if (!empty($tags)): ?>
                                <div class="tags">
                                    <strong>Tags:</strong>
                                    <?php foreach ($tags as $tag): ?>
                                        <span class="tag">
                                            <a href="<?= $prettyUrls ? '/tag/' . $tag['slug'] : '?action=tag&id=' . $tag['id'] ?>">
                                                <?= htmlspecialchars($tag['name']) ?>
                                            </a>
                                        </span>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </article>
                    </div>
                    
                    <footer class="footer">
                        <p><?= htmlspecialchars($settings['footer_text']) ?></p>
                        <?php if ($settings['icp_number']): ?>
                            <p>ICP: <?= htmlspecialchars($settings['icp_number']) ?></p>
                        <?php endif; ?>
                    </footer>
                </body>
                </html>
                <?php
                break;
                
            case 'category':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title><?= htmlspecialchars($category['name']) ?> - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <meta name="description" content="<?= htmlspecialchars($category['description'] ?? $settings['site_description']) ?>">
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        .post { margin-bottom: 40px; }
                        .post-title { font-size: 24px; margin-bottom: 5px; }
                        .post-meta { color: #666; font-size: 14px; margin-bottom: 10px; }
                        .post-excerpt { margin-bottom: 10px; }
                        .read-more { color: #333; font-weight: bold; }
                        .sidebar { float: right; width: 200px; margin-left: 20px; }
                        .category-list, .tag-list { margin-bottom: 20px; }
                        .footer { margin-top: 40px; border-top: 1px solid #eee; padding-top: 20px; }
                    </style>
                </head>
                <body>
                    <header>
                        <h1><a href="<?= $prettyUrls ? '/' : '?' ?>"><?= htmlspecialchars($settings['site_title']) ?></a></h1>
                        <p><a href="?admin">Admin</a></p>
                    </header>
                    
                    <div class="sidebar">
                        <h3>Categories</h3>
                        <ul class="category-list">
                            <?php foreach ($categories as $cat): ?>
                                <li>
                                    <a href="<?= $prettyUrls ? '/category/' . $cat['slug'] : '?action=category&id=' . $cat['id'] ?>">
                                        <?= htmlspecialchars($cat['name']) ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div class="content">
                        <h2>Category: <?= htmlspecialchars($category['name']) ?></h2>
                        <?php if ($category['description']): ?>
                            <p><?= htmlspecialchars($category['description']) ?></p>
                        <?php endif; ?>
                        
                        <?php foreach ($posts as $post): ?>
                            <article class="post">
                                <h3 class="post-title">
                                    <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>">
                                        <?= htmlspecialchars($post['title']) ?>
                                    </a>
                                </h3>
                                <div class="post-meta">
                                    Posted on <?= date('F j, Y', strtotime($post['published_at'])) ?>
                                </div>
                                <div class="post-excerpt">
                                    <?= htmlspecialchars($post['excerpt']) ?>
                                </div>
                                <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>" class="read-more">Read more →</a>
                            </article>
                        <?php endforeach; ?>
                    </div>
                    
                    <footer class="footer">
                        <p><?= htmlspecialchars($settings['footer_text']) ?></p>
                        <?php if ($settings['icp_number']): ?>
                            <p>ICP: <?= htmlspecialchars($settings['icp_number']) ?></p>
                        <?php endif; ?>
                    </footer>
                </body>
                </html>
                <?php
                break;
                
            case 'tag':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Tag: <?= htmlspecialchars($tag['name']) ?> - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
                        .post { margin-bottom: 40px; }
                        .post-title { font-size: 24px; margin-bottom: 5px; }
                        .post-meta { color: #666; font-size: 14px; margin-bottom: 10px; }
                        .post-excerpt { margin-bottom: 10px; }
                        .read-more { color: #333; font-weight: bold; }
                        .sidebar { float: right; width: 200px; margin-left: 20px; }
                        .category-list, .tag-list { margin-bottom: 20px; }
                        .footer { margin-top: 40px; border-top: 1px solid #eee; padding-top: 20px; }
                    </style>
                </head>
                <body>
                    <header>
                        <h1><a href="<?= $prettyUrls ? '/' : '?' ?>"><?= htmlspecialchars($settings['site_title']) ?></a></h1>
                        <p><a href="?admin">Admin</a></p>
                    </header>
                    
                    <div class="sidebar">
                        <h3>Categories</h3>
                        <ul class="category-list">
                            <?php foreach ($categories as $cat): ?>
                                <li>
                                    <a href="<?= $prettyUrls ? '/category/' . $cat['slug'] : '?action=category&id=' . $cat['id'] ?>">
                                        <?= htmlspecialchars($cat['name']) ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    
                    <div class="content">
                        <h2>Tag: <?= htmlspecialchars($tag['name']) ?></h2>
                        
                        <?php foreach ($posts as $post): ?>
                            <article class="post">
                                <h3 class="post-title">
                                    <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>">
                                        <?= htmlspecialchars($post['title']) ?>
                                    </a>
                                </h3>
                                <div class="post-meta">
                                    Posted on <?= date('F j, Y', strtotime($post['published_at'])) ?>
                                </div>
                                <div class="post-excerpt">
                                    <?= htmlspecialchars($post['excerpt']) ?>
                                </div>
                                <a href="<?= $prettyUrls ? '/post/' . $post['slug'] : '?action=post&id=' . $post['id'] ?>" class="read-more">Read more →</a>
                            </article>
                        <?php endforeach; ?>
                    </div>
                    
                    <footer class="footer">
                        <p><?= htmlspecialchars($settings['footer_text']) ?></p>
                        <?php if ($settings['icp_number']): ?>
                            <p>ICP: <?= htmlspecialchars($settings['icp_number']) ?></p>
                        <?php endif; ?>
                    </footer>
                </body>
                </html>
                <?php
                break;
                
            case '404':
                ?>
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Page Not Found - <?= htmlspecialchars($settings['site_title']) ?></title>
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; text-align: center; }
                    </style>
                </head>
                <body>
                    <h1>404 - Page Not Found</h1>
                    <p>The page you requested could not be found.</p>
                    <p><a href="<?= $prettyUrls ? '/' : '?' ?>">Return to homepage</a></p>
                </body>
                </html>
                <?php
                break;
        }
        
        return ob_get_clean();
    }
}